<?php

namespace App\Http\Controllers;
use App\Models\Postulante;
use Illuminate\Http\Request;
use App\Models\Usuario;
use App\Models\Role;
use App\Models\UsuarioRol;
use App\Models\Colegio;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use App\Models\OfertaLaboral;
use Illuminate\Support\Facades\Auth;
use App\Models\PlanEstudio;
use App\Models\Periodo;
use App\Models\Matricula;
use App\Models\Horario;
use Illuminate\Support\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\Curso;
use App\Models\NotaFinal;
use App\Models\NotaLogro;
use App\Models\NotaSubcomponente;
use App\Models\LogroCurso;
use App\Models\MatriculaCurso;
use App\Models\Asistencia;
use App\Models\Instituto;
use Illuminate\Validation\Rule;
class PostulanteController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $postulantes = Postulante::where('grado', 'Estudiante')->get();
    //dd($postulantes); // esto detiene la ejecución y te muestra el contenido

        return view('postulantess.index', compact('postulantes'));
    }


    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        //$colegios = Colegio::all();
        //$mediosPago = MedioPago::all();

        return view('estudiantes.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store1(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'dni' => [
                'required',
                'digits:8',
                'unique:postulantes,dni',
                'unique:usuarios,username' // también valida que no exista como usuario
            ],
            'nombres' => 'required|string|max:255',
            'apellidos' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'foto_postulante' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'certificado_estudios' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // 1. Crear el usuario con el dni como username y contraseña
        $usuario = Usuario::create([
            'username' => $request->dni,
            'password_hash' => bcrypt($request->dni),
            'status' => 1
        ]);

        // 2. Obtener el ID del rol Estudiante
        $rolEstudiante = Role::where('nombre', 'Estudiante')->first();

        if (!$rolEstudiante) {
            return response()->json(['error' => 'Rol "Estudiante" no encontrado'], 500);
        }

        // 3. Asignar el rol al usuario
        UsuarioRol::create([
            'usuario_id' => $usuario->id,
            'rol_id' => $rolEstudiante->id
        ]);

        // 4. Manejar subida de archivos a public/
        $fotoPath = null;
        if ($request->hasFile('foto_postulante')) {
            $foto = $request->file('foto_postulante');
            $fotoPath = 'postulantes/fotos/' . Str::random(10) . '.' . $foto->getClientOriginalExtension();
            $foto->move(public_path('postulantes/fotos'), basename($fotoPath));
        }

        $certificadoPath = null;
        if ($request->hasFile('certificado_estudios')) {
            $certificado = $request->file('certificado_estudios');
            $certificadoPath = 'postulantes/certificados/' . Str::random(10) . '.' . $certificado->getClientOriginalExtension();
            $certificado->move(public_path('postulantes/certificados'), basename($certificadoPath));
        }

        // 5. Crear el postulante
        $postulante = Postulante::create([
            'dni' => $request->dni,
            'nombres' => $request->nombres,
            'apellidos' => $request->apellidos,
            'email' => $request->email,
            'telefono' => $request->telefono,
            'fecha_nacimiento' => $request->fecha_nacimiento,
            'genero' => $request->genero,
            'fuente_inscripcion' => $request->fuente_inscripcion,
            'lengua_materna' => $request->lengua_materna,
            'año_egreso' => $request->año_egreso,
            'discapacidad' => $request->discapacidad,
            'nombre_discapacidad' => $request->nombre_discapacidad,
            'foto_postulante' => $fotoPath,
            'certificado_estudios' => $certificadoPath,
            'usuario_id' => $usuario->id
        ]);

        return response()->json(['success' => 'Postulante registrado correctamente.']);
    }

    public function store(Request $request, \App\Services\MoodleService $moodle)
{
    $validator = Validator::make($request->all(), [
        'dni' => [
            'required',
            'digits:8',
            'unique:postulantes,dni',
            'unique:usuarios,username'
        ],
        'nombres' => 'required|string|max:255',
        'apellidos' => 'required|string|max:255',
        'email' => 'required|email|max:255',
        'foto_postulante' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        'certificado_estudios' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:4096'
    ]);

    if ($validator->fails()) {
        return response()->json(['errors' => $validator->errors()], 422);
    }

    // ======================================================
    // 1) CREAR USUARIO LOCAL
    // ======================================================
    $usuario = Usuario::create([
        'username'      => $request->dni,
        'password_hash' => bcrypt($request->dni),
        'status'        => 1
    ]);

    // Rol estudiante
    $rolEstudiante = Role::where('nombre', 'Estudiante')->first();
    if (!$rolEstudiante) {
        return response()->json(['error' => 'Rol "Estudiante" no encontrado'], 500);
    }

    UsuarioRol::create([
        'usuario_id' => $usuario->id,
        'rol_id'     => $rolEstudiante->id
    ]);

    // ======================================================
    // 2) MANEJO DE ARCHIVOS
    // ======================================================
    $fotoPath = null;
    if ($request->hasFile('foto_postulante')) {
        $foto = $request->file('foto_postulante');
        $fotoPath = 'postulantes/fotos/' . Str::random(10) . '.' . $foto->getClientOriginalExtension();
        $foto->move(public_path('postulantes/fotos'), basename($fotoPath));
    }

    $certificadoPath = null;
    if ($request->hasFile('certificado_estudios')) {
        $certificado = $request->file('certificado_estudios');
        $certificadoPath = 'postulantes/certificados/' . Str::random(10) . '.' . $certificado->getClientOriginalExtension();
        $certificado->move(public_path('postulantes/certificados'), basename($certificadoPath));
    }

    // ======================================================
    // 3) CREAR POSTULANTE
    // ======================================================
    $postulante = Postulante::create([
        'codigo_postulante'   => $request->dni, //  <<< AQUI SE AGREGA
        'dni'                 => $request->dni,
        'nombres'             => $request->nombres,
        'apellidos'           => $request->apellidos,
        'email'               => $request->email,
        'telefono'            => $request->telefono,
        'fecha_nacimiento'    => $request->fecha_nacimiento,
        'genero'              => $request->genero,
        'fuente_inscripcion'  => $request->fuente_inscripcion,
        'lengua_materna'      => $request->lengua_materna,
        'año_egreso'          => $request->año_egreso,
        'discapacidad'        => $request->discapacidad,
        'nombre_discapacidad' => $request->nombre_discapacidad,
        'foto_postulante'     => $fotoPath,
        'certificado_estudios'=> $certificadoPath,
        'usuario_id'          => $usuario->id,
        'grado'               => 'Estudiante' // valor por defecto
    ]);

    // ======================================================
    // 4) CREAR USUARIO EN MOODLE
    // ======================================================
    $passwordMoodle = "Alumno@" . $request->dni;

    $moodleResponse = $moodle->crearUsuario(
        $usuario->username,
        $passwordMoodle,
        $postulante->nombres,
        $postulante->apellidos,
        $postulante->email
    );

    if (isset($moodleResponse[0]['id'])) {
        $usuario->moodle_user_id = $moodleResponse[0]['id'];
        $usuario->save();
    }

    return response()->json([
        'success' => true,
        'message' => 'Postulante registrado y sincronizado con Moodle.',
        'moodle'  => $moodleResponse
    ]);
}


    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $postulante = Postulante::findOrFail($id);
        return view('estudiantes.show', compact('postulante'));
    }


    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
   public function update(Request $request, string $id)
{
    $postulante = Postulante::with('usuario')
        ->where('id_postulante', $id)
        ->firstOrFail();

    // ✅ Validaciones
    $validated = $request->validate([
        'nombres'   => 'required|string|max:100',
        'apellidos' => 'required|string|max:100',
        'dni'       => [
            'required',
            'string',
            Rule::unique('postulantes', 'dni')
                ->ignore($postulante->id_postulante, 'id_postulante'),
        ],
        'email'     => 'required|email|max:100',
        'telefono'  => 'nullable|string|max:15',
        'direccion' => 'nullable|string|max:100',
        'password'  => 'nullable|confirmed|min:6',
    ]);

    // 🔹 ACTUALIZAR POSTULANTE
    $postulante->update([
        'nombres'   => $validated['nombres'],
        'apellidos' => $validated['apellidos'],
        'dni'       => $validated['dni'],
        'email'     => $validated['email'],
        'telefono'  => $validated['telefono'] ?? null,
        'direccion' => $validated['direccion'] ?? null,
    ]);

    // 🔐 ACTUALIZAR CONTRASEÑA EN USUARIOS
    if (!empty($validated['password'])) {

        if ($postulante->usuario) {
            $postulante->usuario->update([
                'password_hash' => Hash::make($validated['password']),
            ]);
        }
    }

    return response()->json([
        'message' => 'Postulante y usuario actualizados correctamente'
    ]);
}

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
{
    $postulante = Postulante::with('matriculas')->findOrFail($id);

    if ($postulante->matriculas->count() > 0) {
        return response()->json([
            'success' => false,
            'message' => 'No se puede eliminar, el postulante tiene matrículas.'
        ], 400);
    }

    $postulante->delete();

    return response()->json([
        'success' => true,
        'message' => 'Postulante eliminado correctamente.'
    ]);
}


    public function perfil()
{
    $usuario = auth()->user();
    $postulante = $usuario->estudiante; // Asegúrate de que esta relación exista

    if (!$postulante) {
        abort(404, 'Perfil de postulante no encontrado.');
    }

    $colegios = Colegio::orderBy('nombre')->get();

    return view('estudiantes.perfil', compact('postulante', 'colegios'));
}


public function verPerfil()
{
    $usuario = auth()->user();
    $postulante = $usuario->estudiante;

    if (!$postulante) {
        abort(404, 'Perfil de postulante no encontrado.');
    }

    $colegios = Colegio::orderBy('nombre')->get();

    return view('estudiantes.perfil_show', compact('postulante', 'colegios'));
}


    public function actualizarPerfil(Request $request)
{
    $usuario = auth()->user();
    $postulante = Postulante::where('usuario_id', $usuario->id)->firstOrFail();

    $request->validate([
        'nombres' => 'required|string|max:100',
        'apellidos' => 'required|string|max:100',
        'dni' => 'required|string|max:15|unique:postulantes,dni,' . $postulante->id_postulante . ',id_postulante',
        'email' => 'nullable|email|max:100',
        'telefono' => 'nullable|string|max:15',
        'genero' => 'nullable|string|max:15',
        'fecha_nacimiento' => 'nullable|date',
        'lengua_materna' => 'nullable|string|max:100',
        'id_colegio' => 'nullable|string|max:100',
        'año_egreso' => 'nullable|string|max:100',
    ]);

    $postulante->update($request->only([
        'nombres', 'apellidos', 'dni', 'email', 'telefono', 'genero',
        'fecha_nacimiento', 'direccion','lengua_materna','id_colegio','año_egreso',
    ]));

    $basePath = public_path('estudiantesdocumentos/' . $postulante->id_postulante);
    if (!file_exists($basePath)) {
        mkdir($basePath, 0755, true);
    }

    // Función para guardar archivos con nombre seguro
    $guardarArchivo = function ($inputName, $carpetaDestino, $atributoModelo) use ($request, $basePath, $postulante) {
        if ($request->hasFile($inputName)) {
            $file = $request->file($inputName);
            $path = $basePath . '/' . $carpetaDestino;
            if (!file_exists($path)) mkdir($path, 0755, true);

            $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $file->getClientOriginalExtension();
            $filename = time() . '_' . \Illuminate\Support\Str::slug($originalName) . '.' . $extension;

            $file->move($path, $filename);
            $postulante->$atributoModelo = 'estudiantesdocumentos/' . $postulante->id_postulante . '/' . $carpetaDestino . '/' . $filename;
        }
    };


    $guardarArchivo('foto_postulante', 'foto', 'foto_postulante');
    $guardarArchivo('certificado_estudios', 'certificados', 'certificado_estudios');
    $guardarArchivo('partida_nacimiento', 'partida', 'partida_nacimiento');
    $guardarArchivo('copia_dni', 'dni', 'copia_dni');
    $guardarArchivo('comprobante_pago', 'pagos', 'comprobante_pago');
    $guardarArchivo('curriculum_archivo', 'curriculum', 'curriculum_archivo');

    $postulante->save();

    return response()->json(['success' => true]);
}
public function ofertasLaborales(Request $request)
{
    // Obtener filtros desde la petición AJAX o inicial
    $nombre = $request->input('nombre');
    $lugar = $request->input('lugar');
    $plan = $request->input('plan');

    // Obtener al postulante autenticado y su plan de estudio
    $postulante = Auth::user()->postulante ?? null;
    $idPlanEstudioPostulante = $postulante->id_plan_estudio ?? null;

    // Construir consulta base con relaciones y estado aceptado
    $query = OfertaLaboral::with(['empresa', 'tipoContrato', 'planEstudio'])
                ->where('estado', 'aceptado');

    // Filtrar por plan de estudio del postulante si aplica (opcional)
    if ($idPlanEstudioPostulante) {
        $query->where('id_plan_estudio', $idPlanEstudioPostulante);
    }

    // Aplicar filtros AJAX si existen
    if ($nombre) {
        $query->where('titulo', 'like', "%{$nombre}%");
    }
    if ($lugar) {
        $query->where('lugar', 'like', "%{$lugar}%");
    }
    if ($plan) {
        $query->where('id_plan_estudio', $plan);
    }

    // Obtener resultados ordenados
    $ofertas = $query->orderByDesc('fecha_publicacion')->get();

    // Cargar planes de estudio para el filtro (solo si no es AJAX)
    if (!$request->ajax()) {
        $planesEstudio = PlanEstudio::orderBy('nombre')->get();

        return view('estudiantes.ofertas.index', compact('ofertas', 'planesEstudio'));
    }

    // Si es AJAX devolver solo la vista parcial con las ofertas filtradas
    return view('estudiantes.ofertas.partials.lista_ofertas', compact('ofertas'))->render();
}

public function obtenerModalOferta(Request $request)
{
    $oferta = OfertaLaboral::with(['empresa', 'tipoContrato'])->findOrFail($request->id_oferta);
    $postulante = Auth::user()->estudiante; // Obtiene el postulante autenticado

    return view('estudiantes.ofertas.partials.modal_oferta', compact('oferta', 'postulante'));
}

public function verHorario()
{
    $postulanteId = Auth::user()->estudiante->id_postulante;

    $matricula = Matricula::where('postulante_id', $postulanteId)
        ->whereHas('periodo', fn($q) => $q->where('activo', '1'))
        ->latest('fecha_matricula')
        ->first();

    if (!$matricula) {
        return view('estudiantes.horario', [
            'horariosDias' => [],
            'horas' => [],
            'dias' => ['Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes'],
            'mensaje' => 'No tienes matrícula activa.'
        ]);
    }

    $cursos = $matricula->matriculaCursos->pluck('curso_id')->toArray();

    $horarios = Horario::whereIn('id_curso', $cursos)
        ->where('id_periodo', $matricula->periodo_id)
        ->get();

    $dias = ['Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes'];

    $horariosPorDia = [];
    $horas = [];

    foreach ($horarios as $h) {
        $franjas = $this->generarFranjasHorarias($h->hora_inicio, $h->hora_fin);

        foreach ($franjas as $f) {
            $horariosPorDia[$h->dia][$f['inicio']] = [
                'curso' => $h->curso->nombre,
                'aula' => $h->numero_aula,
                'fin' => $f['fin']
            ];

            $horas[$f['inicio']] = $f['fin'];
        }
    }

    ksort($horas);

    return view('estudiantes.horario', [
        'horariosDias' => $horariosPorDia,
        'horas' => $horas,
        'dias' => $dias,
        'mensaje' => null
    ]);
}


private function generarFranjasHorarias($inicio, $fin)
{
    $inicio = strtotime($inicio);
    $fin = strtotime($fin);
    $franjas = [];

    while ($inicio < $fin) {
        $proxima = $inicio + 45 * 60;
        $franjas[] = [
            'inicio' => date('H:i', $inicio),
            'fin' => date('H:i', $proxima)
        ];
        $inicio = $proxima;
    }

    return $franjas;
}

public function imprimirHorario()
{
    $postulanteId = Auth::user()->estudiante->id_postulante;

    $matricula = Matricula::where('postulante_id', $postulanteId)
        ->whereHas('periodo', fn($q) => $q->where('activo', '1'))
        ->latest('fecha_matricula')
        ->first();

    if (!$matricula) {
        return redirect()->back()->with('error', 'No tienes matrícula activa.');
    }

    // === CONSULTAR INSTITUTO ===
    // Si no existe, evita errores
    $instituto = Instituto::first() ?? null;

    $cursos = $matricula->matriculaCursos->pluck('curso_id')->toArray();

    $horarios = Horario::whereIn('id_curso', $cursos)
        ->where('id_periodo', $matricula->periodo_id)
        ->get();

    $dias = ['Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes'];
    $horariosPorDia = [];
    $horas = [];

    foreach ($horarios as $h) {
        $franjas = $this->generarFranjasHorarias($h->hora_inicio, $h->hora_fin);

        foreach ($franjas as $f) {
            $horariosPorDia[$h->dia][$f['inicio']] = [
                'curso' => $h->curso->nombre,
                'aula' => $h->numero_aula,
                'fin'   => $f['fin']
            ];

            $horas[$f['inicio']] = $f['fin'];
        }
    }

    // Ordenar por hora
    ksort($horas);

    $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('estudiantes.horario_pdf', [
        'horariosDias' => $horariosPorDia,
        'horas'        => $horas,
        'dias'         => $dias,
        'estudiante'   => Auth::user()->estudiante,
        'instituto'    => $instituto
    ]);

    return $pdf->stream('horario.pdf');
}


public function verNotas($cursoId)
{
    $estudianteId = Auth::user()->estudiante->id_postulante;
    

    $curso = Curso::findOrFail($cursoId);
    $notaFinal = NotaFinal::where('estudiante_id', $estudianteId)->where('curso_id', $cursoId)->first();

    $logros = LogroCurso::with('subcomponentes')->where('curso_id', $cursoId)->get();

    return view('estudiantes.partials.notas_modal', [
        'curso' => $curso,
        'notaFinal' => $notaFinal,
        'logros' => $logros,
        'estudianteId' => $estudianteId
    ]);
}
public function verAsistencia($cursoId)
{
    $postulanteId = Auth::user()->estudiante->id_postulante;

    $matriculaCurso = MatriculaCurso::whereHas('matricula', function ($q) use ($postulanteId) {
        $q->where('postulante_id', $postulanteId)
          ->whereHas('periodo', fn($p) => $p->where('activo', '1'));
    })->where('curso_id', $cursoId)->first();

    if (!$matriculaCurso) {
        return response()->json(['html' => '<p class="text-danger">No se encontró asistencia.</p>']);
    }

    $asistencias = Asistencia::with('sesion')
        ->where('matricula_curso_id', $matriculaCurso->id)
        ->orderBy('fecha')
        ->get();

    $html = view('estudiantes.partials.modal_asistencia', compact('asistencias'))->render();

    return response()->json(['html' => $html]);
}
public function historialAcademico()
{
    $estudianteId = Auth::user()->estudiante->id_postulante;

    // Todas las matrículas ordenadas desde la más antigua
    $matriculas = Matricula::with([
        'periodo',
        'semestre',
        'matriculaCursos.curso',
        'matriculaCursos.curso.modulo', // si necesitas módulos formativos
        'matriculaCursos.curso.notaFinal' => function($q) use($estudianteId) {
            $q->where('estudiante_id', $estudianteId);
        }
    ])
    ->where('postulante_id', $estudianteId)
    ->orderBy('fecha_matricula', 'asc')
    ->get();

    return view('estudiantes.historial', compact('matriculas'));
}

public function importarExcel(Request $request)
{
    $request->validate([
        'archivo' => 'required|file|mimes:xlsx,xls'
    ]);

    try {

        \Excel::import(new \App\Imports\EstudiantesMasivoImport, $request->file('archivo'));

        return response()->json([
            'success' => true,
            'message' => 'Los estudiantes fueron importados correctamente.'
        ]);

    } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {

        $errores = [];
        foreach ($e->failures() as $failure) {
            $errores[] = "Fila {$failure->row()}: " . implode(", ", $failure->errors());
        }

        return response()->json(['errors' => $errores], 422);

    } catch (\Exception $e) {
        return response()->json([
            'message' => "Error inesperado: " . $e->getMessage()
        ], 500);
    }
}

}
